#include "RatioTemplate.h"

#include <vector>
#include <iostream>
#include <fstream>
#include <cmath>
#include <stdexcept>
using namespace std;

RatioTemplate::RatioTemplate()
 : regions_( NULL ),
   used_( NULL ),
   numRegions_( 0 ),
   ratios_( NULL ),
   numRatios_( 0 )
{

}

RatioTemplate::~RatioTemplate() {
	if ( regions_ ) delete[] regions_;
	regions_ = NULL;
	if ( used_ ) delete[] used_;
	used_ = NULL;
	if ( ratios_ ) delete[] ratios_;
	ratios_ = NULL;
}
	
void RatioTemplate::setNumRegions( int num ) {
	//cout << "setNumRegions " << num << endl;
	if ( num == numRegions_ ) return;
	//cout << "setNumRegions " << num << endl;
	if ( regions_ ) delete[] regions_;
	regions_ = new Region[ num ];
	if ( used_ ) delete[] used_;
	used_ = new bool[ num ];
	numRegions_ = num;
	resetUsedRegions();
}
	
void RatioTemplate::setNumRatios( int num ) {
	if ( num == numRatios_ ) return;
	
	if ( ratios_ ) delete[] ratios_;
	ratios_ = new Ratio[ num ];
	numRatios_ = num;
}
	
void RatioTemplate::setRegion( int index, double x, double y, double width, double height ) {
	if ( index < 0 || index >= numRegions_ )
		throw runtime_error( "index out of bounds" );
	//cout << "cpp: " << x << " " << y << " " << width << " " << height << endl;
	Region& region = regions_[ index ];
	region.x = x;
	region.y = y;
	region.width  = width;
	region.height = height;
	setImageSize( 20, 20 );
}

void RatioTemplate::setRatio( int index, double threshold, int region1, int region2, double importance ) {
	if ( index < 0 || index >= numRatios_ )
		throw runtime_error( "index out of bounds" );
	Ratio& ratio = ratios_[ index ];
	ratio.threshold  = threshold;
	ratio.region1    = region1;
	ratio.region2    = region2;
	ratio.importance = importance;
	used_[ region1 ] = used_[ region2 ] = true;
}

void RatioTemplate::resetUsedRegions() {
	assert( used_ || numRegions_ == 0 );
	for ( int i = 0; i < numRegions_; i++ ) used_[ i ] = false;
}

void RatioTemplate::setImageSize( int width, int height ) {
	for ( int i = 0; i < numRegions_; i++ ) {
		Region& region = regions_[ i ];
		int x1 = (int)floor(region.x*width+0.5);
		int y1 = (int)floor(region.y*height+0.5);
		int x2 = (int)floor((region.x+region.width)*width+0.5);
		int y2 = (int)floor((region.y+region.height)*height+0.5);
		x1 = max( 0, x1 );
		y1 = max( 0, y1 );
		x2 = min( width, x2 );
		y2 = min( height, y2 );
		region.ix = x1;
		region.iy = y1;
		region.iwidth  = x2-x1;
		region.iheight = y2-y1;
	}
}

bool RatioTemplate::matches( const Image& img ) const {
	return matches( img.image(), img.width(), img.height() );
}
	
bool RatioTemplate::matches( const CvArr* image, int width, int height ) const {
	//int width  = img.width();
	//int height = img.height();
	assert( width == 20 && height == 20 );
	//Image normalised( width, height );
	//normalised.copy( img ).normalise();
	//const CvArr* image = normalised.image();
	//const CvArr* image = img.image();
	
	vector<double> regionValues( numRegions_ );
	CvMat mat;
	
	//cout << "numRegions_ " << numRegions_ << endl;
	assert( used_ );
	
	for ( int i = 0; i < numRegions_; i++ ) {
		if ( !used_[ i ] )
			continue;
		Region& region = regions_[ i ];
		int x = region.ix;
		int y = region.iy;
		int w = region.iwidth;
		int h = region.iheight;
		//cout << x << " " << y << " " << w << " " << h << endl;
		if ( w > 0 && h > 0 ) {
			CvRect rect = cvRect( x, y, w, h );
			cvGetSubRect( image, &mat, rect );
			CvScalar avg = cvAvg( &mat );
			regionValues[ i ] = avg.val[ 0 ];
			//cout << avg.val[0] << endl;
		}
		else {
			regionValues[ i ] = 0.0;
		}
	}
	
	double matching = 0;
	for ( int i = 0; i < numRatios_; i++ ) {
		Ratio& ratio = ratios_[ i ];
		double val1  = regionValues[ ratio.region1 ];
		double val2  = regionValues[ ratio.region2 ];
		if ( val2 == 0.0 ) // avoid division by zero
			val2 = 1.0/512.0; // equivalent to half a pixel
			
		if ( (val1/val2) > ratio.threshold )
			matching += ratio.importance;
		else
			matching -= ratio.importance;
	}
	
	return matching >= 0.0;
}

void RatioTemplate::draw( Image& img ) const {
	int width  = img.width();
	int height = img.height();
	
	double color = 0.0;
	for ( int i = 0; i < numRegions_; i++ ) {
		if ( !used_[ i ] )
			continue;
		Region& region = regions_[ i ];
		double x = region.x;
		double y = region.y;
		double w = region.width;
		double h = region.height;
		int x1 = (int)floor(x*width+0.5),  x2 = (int)floor((x+w)*width+0.5);
		int y1 = (int)floor(y*height+0.5), y2 = (int)floor((y+h)*height+0.5);
		img.line( color, x1, y1, x2, y1 );
		img.line( color, x2, y1, x2, y2 );
		img.line( color, x2, y2, x1, y2 );
		img.line( color, x1, y2, x1, y1 );
	}
	
	for ( int i = 0; i < numRatios_; i++ ) {
		Ratio& ratio = ratios_[ i ];
		Region& region1 = regions_[ ratio.region1 ];
		Region& region2 = regions_[ ratio.region2 ];
		int x1 = (int)(width*(region1.x + 0.5*region1.width));
		int y1 = (int)(height*(region1.y + 0.5*region1.height));
		int x2 = (int)(width*(region2.x + 0.5*region2.width));
		int y2 = (int)(height*(region2.y + 0.5*region2.height));
		
		img.line( color, x2, y2, x1, y1 );
		
		if ( x1 == x2 && y1 == y2 ) // pointing at same place
			continue;
		
		// now draw arrow head
		double dx = x2-x1, dy = y2-y1;
		double dist = sqrt( dx*dx + dy*dy );
		dx = dx/dist;
		dy = dy/dist;
		
		int len=5, w = 3;
		int x3 = (int)(x1 +(len*dx - w*dy));
		int y3 = (int)(y1 +(len*dy + w*dx));
		int x4 = (int)(x1 +(len*dx + w*dy));
		int y4 = (int)(y1 +(len*dy - w*dx));
		img.line( color, x1, y1, x3, y3 );
		img.line( color, x3, y3, x4, y4 );
		img.line( color, x4, y4, x1, y1 );
		
		//char label[ 20 ];
		//sprintf( label, "r%d", i );
		//img.text( label, color, (x1+x2)/2, (y1+y2)/2 );
	}
}

void RatioTemplate::save( const char* name ) const {
	ofstream out( name );
	out << numRegions_ << endl;
	for ( int i = 0; i < numRegions_; i++ ) {
		Region& region = regions_[ i ];
		out << region.x << " " << region.y << " " << region.width << " " << region.height << endl;
	}
	out << endl;
	out << numRatios_ << endl;
	for ( int i = 0; i < numRatios_; i++ ) {
		Ratio& ratio = ratios_[ i ];
		out << ratio.threshold << " " << ratio.region1 << " " << ratio.region2 << " " << ratio.importance << endl;
	}
	out << endl;
	out.close();
}
	
void RatioTemplate::load( const char* name ) {

	ifstream in( name );
	
	int num = 0;
	in >> num;
	setNumRegions( num );
	for ( int i = 0; i < numRegions_; i++ ) {
		Region& r = regions_[ i ];
		in >> r.x >> r.y >> r.width >> r.height;
	}
	
	in >> num;
	setNumRatios( num );
	for ( int i = 0; i < numRatios_; i++ ) {
		Ratio& r = ratios_[ i ];
		in >> r.threshold >> r.region1 >> r.region2 >> r.importance;
		used_[ r.region1 ] = used_[ r.region2 ] = true;
	}
	setImageSize(20,20);
	in.close();
}

