#include "Image.h"

#include "rand-util.h"

#include "opencv/cv.h"
#include "opencv/highgui.h"
//#include "opencv/cverror.h"

#include <iostream>
#include <stdexcept>
using namespace std;

//int CV_CDECL MyErrorCallBack(CVStatus status, const char *func,const char *context, const char *file,int line) {
//	assert( false );
//	return 0;
//}

//unsigned int Image::DEPTH = IPL_DEPTH_32F;
unsigned int Image::DEPTH = IPL_DEPTH_8U;

Image::Image()
 : image_( NULL ),
   w_( -1 ),
   h_( -1 )
{

}

Image::Image( int w, int h, int depth )
 : image_( NULL ),
   w_( w ),
   h_( h ),
   type_( IPLIMAGE )
{
	CvSize imgSize;
	imgSize.width = w;
	imgSize.height = h;
	image_ = cvCreateImage( imgSize, depth, 1 );
	set( 0.0 );
}

Image::Image( const Image& img ) 
 : image_( NULL ),
   w_( img.w_ ),
   h_( img.h_ ),
   type_( img.type_ )
{
	switch( type_ ) {
		case IPLIMAGE:
			copy( img );
		break;
		case MATRIX: {
			image_ = (CvMat*)cvAlloc( sizeof( CvMat ) );
			*((CvMat*)image_) = *((CvMat*)img.image_);
		}			
		break;
	}
}
	
Image::Image( const Image& img, int x, int y, int width, int height )
 : w_( width ),
   h_( height ),
   type_( MATRIX )
{

	//cout << x << " " << y << " " << width << " " << height << endl;
	if ( !( x >= 0 && y >= 0 && (x+width) <= img.width() && (y+height) <= img.height() ) )
		cout << x << " " << y << " " << width << " " << height << endl;
	assert(  x >= 0 && y >= 0 && (x+width) <= img.width() && (y+height) <= img.height() );
	image_ = (CvMat*)cvAlloc( sizeof( CvMat ) );
	CvRect rect = cvRect( x, y, width, height );
	cvGetSubRect( img.image_, (CvMat*)image_, rect );
}


Image::Image( const char* filename )
 : image_( NULL ),
   w_( -1 ),
   h_( -1 )
{
	//cvRedirectError( MyErrorCallBack );
	load( filename );
}
	
Image::~Image() {
	release_image();
}
		
void Image::load( const char* filename ) {
	release_image();
	image_ = cvLoadImage( filename, 0 );
	CvSize size;
	size = cvGetSize( image_ );
	w_ = size.width;
	h_ = size.height;
	type_ = IPLIMAGE;
}

void Image::save( const char* filename ) const {
	// has to be 8bit grayscale to save
	cvSaveImage( filename, image_ );
}


double Image::avg() const {
	CvScalar average;
	average = cvAvg( image_ );
	return average.val[ 0 ];
}

double Image::sum() const {
	CvScalar summation;
	summation = cvSum( image_ );
	return summation.val[ 0 ];
}

double Image::moment() const {
	CvMoments moments;
	cvMoments( image_, &moments );
	return cvGetSpatialMoment( &moments, 0, 0 );
}

double Image::momentx() const {
	CvMoments moments;
	cvMoments( image_, &moments );
	return cvGetSpatialMoment( &moments, 1, 0 );
}

double Image::momenty() const {
	CvMoments moments;
	cvMoments( image_, &moments );
	return cvGetSpatialMoment( &moments, 0, 1 );
}

void Image::release_image() {
	switch( type_ ) {
		case IPLIMAGE: 
			cvReleaseImage( (IplImage**)&image_ );
		break;
		case MATRIX:
			cvFree( &image_ );
		break;
	}
}

Image Image::subimage( int x, int y, int width, int height ) const {
	//cout << x << " " << y << " " << width << " " << height << endl;
	Image img( *this, x, y, width, height );
	return img;	
}

// mutators

Image& Image::set( double value ) {
	CvScalar scalar;
	scalar = cvRealScalar( value );
	cvSet( image_, scalar );
	return *this;
}

Image& Image::copy( const Image& rhs ) {
	return copyCvArr( rhs.image() );
}

Image& Image::copyCvArr( const CvArr* image ) {
	if ( image_ == image )
		return *this;
	
	CvSize size = cvGetSize( image );
	if ( size.width < 0 || size.height < 0 )
		return *this;
		
	if ( w_ != size.width || h_ != size.height || image_ == NULL ) {
		w_ = size.width;
		h_ = size.height;
		release_image();
		image_ = cvCreateImage( size, DEPTH, 1 );
		type_ = IPLIMAGE;	
	}
	cvCopy( image, image_ );
	return *this;
}

Image& Image::diff( const Image& img ) {
	cvAbsDiff( image_, img.image_, image_ );
	return *this;
}

Image& Image::blur() {
	cvSmooth( image_, image_, CV_GAUSSIAN, 3, 3 );
	return *this;
}

Image& Image::resize( int width, int height ) {
	if ( width == w_ && height == h_ )
		return *this;
	CvSize size;
	size = cvSize( width, height );
	IplImage* image = cvCreateImage( size, DEPTH, 1 );
	//cvResize( image_, image, CV_INTER_NN );
	cvResize( image_, image );
	release_image();
	image_ = image;
	w_ = width;
	h_ = height;
	type_ = IPLIMAGE;
	return *this;
}

Image& Image::scalePyramidDown() {
	w_ = w_/2;
	h_ = h_/2;
	CvSize size;
	size = cvSize( w_, h_ );
	IplImage* image = cvCreateImage( size, DEPTH, 1 );
	cvPyrDown( image_, image );
	release_image();
	image_ = image;
	type_ = IPLIMAGE;
	return *this;
}

Image& Image::scalePyramidUp() {
	w_ = w_*2;
	h_ = h_*2;
	CvSize size;
	size = cvSize( w_, h_ );
	IplImage* image = cvCreateImage( size, DEPTH, 1 );
	cvPyrUp( image_, image );
	release_image();
	image_ = image;
	type_ = IPLIMAGE;
	return *this;
}

Image& Image::normalise() {
	CvPoint minLoc, maxLoc;
	double min = 0, max = 0;
	cvMinMaxLoc( image_, &min, &max, &minLoc, &maxLoc );
	cvSubS( image_, cvRealScalar(min), image_ );
	if ( min != max )
		cvConvertScale( image_, image_, 1.0/(max-min) );
	return *this;
}

void Image::line( double value, int x1, int y1, int x2, int y2 ) {
	CvPoint p1;
	CvPoint p2;
	p1.x = x1;
	p1.y = y1;
	p2.x = x2;
	p2.y = y2;
	cvLine( image_, p1, p2, (value) );
}

void Image::rect( double value, int x1, int y1, int x2, int y2 ) {
	//assert( x2 < w_ && y2 < h_ );
	CvPoint p1;
	CvPoint p2;
	p1.x = x1;
	p1.y = y1;
	p2.x = x2;
	p2.y = y2;
	cvRectangle( image_, p1, p2, (value), 1 );
}

void Image::text( const char* text, double color, int x, int y ) {
	CvFont font;
	double size = 0.25;
	cvInitFont( &font, CV_FONT_VECTOR0, size, size, 0, 1 );
	CvPoint pt = cvPoint( x, y );
	cvPutText( image_, text, pt, &font, (int)(color) );
}
