import Project
from random import choice,uniform,gauss


class Candidate:
	def __init__(self,fitnessFunction,mutationSize,regions):
		self.fitnessFunction=fitnessFunction
		self.mutationSize=mutationSize
		self.template=Project.RatioTemplate()
		self.template.setNumRegions(len(regions))
		for index in range( 0, len( regions ) ):
			region=regions[index]
			self.template.setRegion(index,region[0],region[1],region[2],region[3])
		self.invalidate()
	
	# implement in sub-classes
	def initialiseTemplate(self,template):
		pass
	
	# initialise the template (if needed) and return it
	def getTemplate(self):
		if not self.templateValid:
			self.initialiseTemplate(self.template)
			self.templateValid=True
		return self.template
	
	# force a fitness evaluation
	def invalidate(self):
		self.templateValid=False
		self.fitnessValid=False
		self.fitnessValue=-1.0
	
	# return the fitness, re-calculating it if needed
	def fitness(self):
		if not self.fitnessValid:
			self.fitnessValue=self.fitnessFunction.evaluate(self)
			self.fitnessValid=True
		return self.fitnessValue
	
	# copy the other candidate into this one
	def copy(self,other):
		self.mutationSize=other.mutationSize
		self.invalidate()
	
	# mutate real values using a gaussian random number and cap it between 0-1
	def mutateGauss(self,values):
		return map( lambda x: max(0.0,min(1.0,x+self.mutationSize*gauss(0,1))), values )
	
	def pointMutate(self,value,alleles,mutationRate):
		if uniform(0,1) < mutationRate:
			return choice(alleles)
		return value
	
	def mutateAllele(self,values,alleles):
		return map( lambda x: self.pointMutate(x,alleles,1.0/len(values)), values )
	
	# apply a single mutation
	def mutate(self):
		self.invalidate()
	
	# discrete uniform crossover
	def recombine(self,other):
		self.invalidate()
	
	def interRegionDistance(self,r1,r2):
		x1,y1=r1.x+r1.width/2,r1.y+r1.height/2
		x2,y2=r2.x+r2.width/2,r2.y+r2.height/2
		dx,dy=x1-x2,y1-y2
		return dx*dx + dy*dy
	
	# (randomly) select a region that is a "neighbour" of the one given
	def selectNeighbouringRegion(self,regionIndex):
		template=self.getTemplate()
		regionIndices=range(0,template.getNumRegions())
		regionIndices.remove(regionIndex) # so were look at the other regions
		region=template.getRegion(regionIndex)
		regions=[]
		for index in regionIndices:
			regions.append( template.getRegion( index ) )
		
		indicesSorted=[]
		while len(regions) > 0:
			nearest=regions[0]
			nearestIndex=regionIndices[0]
			for r in regions:
				if self.interRegionDistance(region,r) < self.interRegionDistance(region,nearest):
					nearest=r
					nearestIndex=regionIndices[regions.index(r)]
			indicesSorted.append(nearestIndex)
			regions.remove(nearest)
			regionIndices.remove(nearestIndex)
		
		# biased to return nearer neighbours
		return indicesSorted[int(len(indicesSorted)*(uniform(0,1)**2))]
	
	def __str__(self):
		return "Candidate[fitness="+str(self.fitness())+"]"
