/*
 * Simulation.java
 *
 * Created on 20 October 2003, 10:33
 */

package proj1.simulator;

import java.util.*;

import proj1.simulator.physics.Body;
import proj1.simulator.physics.Vector2d;

/**
 *
 * @author  msc37jxm
 */
public class Simulation implements CollisionHandler {
    private float dt = 0.04f;
    private List vehicles = new ArrayList();
	private List objects  = new ArrayList();
    private int width, height;
    private boolean edgesSolid = true;
    private float boundaryReconstitution = 0.9f;
    private float vehicleReconstitution = 0.9f;
    
    private Vector2d leftBoundaryNormal   = new Vector2d( 1, 0 );
	private Vector2d topBoundaryNormal    = new Vector2d( 0, 1 );
	private Vector2d rightBoundaryNormal  = new Vector2d( -1,  0 );
	private Vector2d bottomBoundaryNormal = new Vector2d(  0, -1 );  
    
    private float sensorRange = 32.0f;
	private float sensorRangeSq = sensorRange*sensorRange;
    
    private CollisionTree collisionTree = null;
	private CollisionHandler collisionHandler = this;
	private CollisionController collisionController = null;
    
    private int timeStep = 0;
    
    /** Creates a new instance of Simulation */
    public Simulation( int width, int height ) {
    	this.width = width;
    	this.height = height;
    	// TODO figure out how sensor range and bucket size interact
		collisionTree = new CollisionTree( this, width/10, height/10 );
		setCollisionHandler( this );
    }
    
    public int getWidth() {
    	return width;
    }
    
    public int getHeight() {
    	return height;
    }
    
    public void setCollisionHandler( CollisionHandler collisionHandler ) {
    	this.collisionHandler = collisionHandler;
		collisionController = new CollisionController( this, collisionHandler );
    }
    
    public int getNumVehicles() {
        return vehicles.size();
    }
    
    public Vehicle getVehicle( int i ) {
        return (Vehicle)vehicles.get( i );
    }
    
    public void addVehicle( Vehicle veh ) {
        vehicles.add( veh );
        sensorRange = Math.max( sensorRange, veh.getMaxSensorRange() );
		sensorRangeSq = sensorRange*sensorRange;
    }

	public void removeVehicle( Vehicle veh ) {
        vehicles.remove( veh );
    }
	
	public void removeAllVehicles() {
		vehicles.clear();
	}
	
	public int getNumFixedObjects() {
        return objects.size();
    }
    
    public FixedObject getFixedObject( int i ) {
        return (FixedObject)objects.get( i );
    }
	
	public void addFixedObject( FixedObject obj ) {
        objects.add( obj );
		collisionTree.add(obj);
    }

	public void removeFixedObject( FixedObject obj ) {
        objects.remove( obj );
		collisionTree.remove(obj);
    }
	
    private void checkBoundaryCollisions( List vehicles ) {
    		
    	Vector2d p = new Vector2d();
    	
    	int numVehicles = vehicles.size();
		for ( int i = 0; i < numVehicles; i++ ) {
			Vehicle veh = (Vehicle)vehicles.get( i );
			Vector2d pos = veh.getPosition();
			float radius = veh.getRadius();
			
			// check left
			if ( pos.x-radius <= 0 ) {
				collisionHandler.handleBoundaryCollision( this, veh, p.set( pos.x-radius, pos.y ), leftBoundaryNormal );
			}
			
			// check top
			if ( pos.y-radius <= 0 ) {
				collisionHandler.handleBoundaryCollision( this, veh, p.set( pos.x, pos.y-radius ), topBoundaryNormal );
			}
					 
			// check right
			if ( pos.x+radius >= width ) {
				collisionHandler.handleBoundaryCollision( this, veh, p.set( pos.x+radius, pos.y ), rightBoundaryNormal );
			}
					 
			// check bottom
			if ( pos.y+radius >= height ) {
				collisionHandler.handleBoundaryCollision( this, veh, p.set( pos.x, pos.y+radius ), bottomBoundaryNormal );
			}
		}
    }
    
    public void handleBoundaryCollision( Simulation sim, Vehicle veh, Vector2d pos, Vector2d normal ) {
    	Body body = veh.getBody();
    	Vector2d mom = new Vector2d();
    	
    	mom = body.calculatePointMomentum( pos, mom );
    	
    	// heading away
    	if ( mom.dotProduct( normal ) >= 0 )
    		return;
    	
    	float r = boundaryReconstitution;
    	float impulse = -normal.dotProduct( mom.multiply( 1.0f + r ) );
    	body.applyImpulse( pos.x, pos.y, normal.x*impulse, normal.y*impulse );
    	
    }
    
    public void handleCollision( Simulation sim, Vehicle veh1, Vehicle veh2 ) {
		
    	Vector2d normal = new Vector2d();
    	Vector2d mom    = new Vector2d();
    	
    	Body body1 = veh1.getBody();
    	Body body2 = veh2.getBody();
    	
    	Vector2d pos1 = body1.getPosition();
    	Vector2d pos2 = body2.getPosition();
    	
    	// we will ignore angular effects when colliding
    	Vector2d mom1 = body1.getMomentum();
    	Vector2d mom2 = body2.getMomentum();
    	
    	normal = normal.subtract( pos1, pos2 );
    	mom    = mom.subtract( mom1, mom2 );
    	
    	// TODO apply a small force if touching, but only moving apart slowly
    	// moving apart
    	if ( mom.dotProduct( normal ) >= 0 )
    		return;
    	
    	normal = normal.normalise();
    	
		float r = vehicleReconstitution;
		float i = -normal.dotProduct( mom.multiply( 1.0f + r ) );
    	
    	// impulse is divided between vehicles equally
    	Vector2d impulse = normal.multiply( i*0.5f ); 
    	
		body1.applyImpulse( pos1.x, pos1.y,  impulse.x,  impulse.y );
		body2.applyImpulse( pos2.x, pos2.y, -impulse.x, -impulse.y );
    }
	
	public void handleCollision( Simulation sim, Vehicle veh, FixedObject obj ) {
		
    	Vector2d normal = new Vector2d();
    	Vector2d mom    = new Vector2d();
    	
    	Body body1 = veh.getBody();
    	
    	Vector2d pos1 = body1.getPosition();
    	Vector2d pos2 = obj.getPosition();
    	
    	// we will ignore angular effects when colliding
    	Vector2d mom1 = body1.getMomentum();
    	
    	normal = normal.subtract( pos1, pos2 );
    	mom    = mom.set( mom1 );
    	
    	// TODO apply a small force if touching, but only moving apart slowly
    	// moving apart
    	if ( mom.dotProduct( normal ) >= 0 )
    		return;
    	
    	normal = normal.normalise();
    	
		float r = vehicleReconstitution;
		float i = -normal.dotProduct( mom.multiply( 1.0f + r ) );
    	
    	Vector2d impulse = normal.multiply( i ); 
    	
		body1.applyImpulse( pos1.x, pos1.y,  impulse.x,  impulse.y );
    }
    
	
	public void checkCollisionsAndSensors( List vehicles, List fixedObjects ) {
    	
		// TODO check for repeat collisions
    	int numVehicles = vehicles.size();
    	for ( int i = 0; i < numVehicles; i++ ) {
    		Vehicle vehi = (Vehicle)vehicles.get( i );
    		Vector2d posi = vehi.getPosition();
    		float radiusi = vehi.getRadius();
    		
    		for ( int j = i+1; j < numVehicles; j++ ) {
    			Vehicle vehj = (Vehicle)vehicles.get( j );
    			Vector2d posj = vehj.getPosition();
    			float radiusj = vehj.getRadius();
				//collCount++;
				
				float distSq = posi.distanceSq( posj );
				
				float radius = radiusi+radiusj;
				
    			if ( distSq < (radius*radius) )
					collisionController.vehicleCollision( vehi, vehj );
    				
    			if ( distSq < sensorRangeSq ) {
    				vehi.updateSensors( vehj );
					vehj.updateSensors( vehi );
    			}
    		}
			
			// run backwards so we can safely remove objects
			for ( int j = fixedObjects.size()-1; j >= 0; j-- ) {
				FixedObject obj = (FixedObject)fixedObjects.get( j );
				Vector2d posj = obj.getPosition();
    			float radiusj = obj.getRadius();
				
				float distSq = posi.distanceSq( posj );
				
				float radius = radiusi+radiusj;
				
    			if ( distSq < (radius*radius) )
    				collisionController.fixedObjectCollision( vehi, obj );
    				
    			if ( distSq < sensorRangeSq ) {
    				vehi.updateSensors( obj );
    			}
			}
    	}
    }
    
    private int collCount = 0;
    
    
    public int getTimeStep() {
    	return timeStep;
    }
    
    public void step() {
        
		int numVehicles = vehicles.size();
        for ( int i = 0; i < numVehicles; i++ ) {
            Vehicle veh = (Vehicle)vehicles.get( i );
            veh.applyForces();
        }
        
		collisionTree.add( vehicles );
        
        // we can do this outside the
        // collision tree as detecting
        // boundary collisions is relatively
        // cheap, and the number of tests required
        // varies linearly with pop size
        if ( edgesSolid )
			checkBoundaryCollisions( vehicles );
		
		collCount = 0;
		
		collisionTree.checkCollisionsAndSensors();
		collisionTree.clear();
		
		
		boolean check = false;
		
		if ( check ) {
			int numCollisions = collisionController.getNumCollisions();
			collisionController.reset();
			checkCollisionsAndSensors( vehicles, objects );
			if ( numCollisions != collisionController.getNumCollisions() )
				System.out.println( "number collisions not matching (" + numCollisions + " vs. " + collisionController.getNumCollisions() + ")" );
		}
		
		collisionController.doCollisions();
        
        for ( int i = 0; i < numVehicles; i++ ) {
            Vehicle veh = (Vehicle)vehicles.get( i );
            veh.updateSensors( width, height );
            veh.integrate( dt );
			veh.update();
        }
        
        /*for ( int i = 0; i < numVehicles; i++ ) {
            Vehicle veh = (Vehicle)vehicles.get( i );
            veh.update();
        }*/
		timeStep++;
    }
    
}
