/*
 * Created on Nov 6, 2003
 *
 */
package proj1.gui;

import java.awt.*;
import java.awt.event.*;
import java.text.MessageFormat;
import java.util.zip.GZIPInputStream;

import javax.swing.*;
import java.io.*;

import proj1.*;
import proj1.simulator.Simulation;
import proj1.simulator.Vehicle;

/**
 * @author john
 *
 */
public class MainGUI implements Runnable {
	
	private Environment environment = null;
	
	private JFrame mainFrame = null;
	
	private JMenu fileMenu = null;
		private JMenuItem newSimulationMenuItem = null;
		private JMenuItem savePopulationMenuItem = null;
		private JMenuItem loadPopulationMenuItem = null;
		private JMenuItem quitMenuItem          = null;
	
	private JMenu controlsMenu = null;
		private JMenuItem runMenuItem   = null;
		private JMenuItem pauseMenuItem = null;
		private JMenuItem stepMenuItem  = null;
	
	private JMenu viewMenu = null;
		private JMenuItem zoomInMenuItem  = null;
		private JMenuItem zoomOutMenuItem = null;
		private JCheckBoxMenuItem renderMenuItem = null;
		private JMenuItem statsMenuItem   = null;
	
	private SimPanel simPanel = null;
	private JScrollPane scrollPane = null;
	
	private JLabel timeStepLabel = null;
	private JLabel speedLabel = null;
	private JLabel offspringLabel = null;
	private JLabel fitnessLabel = null;
	private float speed = 0.0f;
	
	private volatile boolean running = false;
	private boolean render = true;
	
	public void init() {
		mainFrame = new JFrame( "Simulation" );
		//mainFrame.setDefaultCloseOperation( JFrame.DISPOSE_ON_CLOSE );
		mainFrame.addWindowListener(
			new WindowAdapter() {
				public void windowClosing( WindowEvent we ) {
					mainFrame.dispose();
					setRunning( false );	
				}
				
				public void windowClosed( WindowEvent we ) {
					setRunning( false );
					try {
						Thread.sleep( 100 );	
					}
					catch( Exception e ){}
					System.exit( 0 );
				}
			}
		);
		
		initMainMenu();
		
		simPanel = new SimPanel();
		scrollPane = new JScrollPane( simPanel );
		
		timeStepLabel = new JLabel( "00" );
		speedLabel    = new JLabel( "x1" );
		offspringLabel = new JLabel( "0 offspring" );
		fitnessLabel   = new JLabel( "0.0" );
		
		JPanel labelPanel = new JPanel( new GridLayout( 1, 4 ) );
		
		labelPanel.add( timeStepLabel );
		labelPanel.add( speedLabel );
		labelPanel.add( offspringLabel );
		labelPanel.add( fitnessLabel );
		
		mainFrame.getContentPane().setLayout( new BorderLayout() );
		mainFrame.getContentPane().add( scrollPane, BorderLayout.CENTER );
		mainFrame.getContentPane().add( labelPanel, BorderLayout.SOUTH );
		
		setRunning( false );
		showTitle();
		
		mainFrame.pack();
		mainFrame.show();
	}
	
	private void setRunning( boolean running ) {
		this.running = running;
		runMenuItem.setEnabled( !running );
		pauseMenuItem.setEnabled( running );
		stepMenuItem.setEnabled( !running );
	}
	
	private void showTitle() {
		int zoom = (int)(simPanel.getScale()*100);
		mainFrame.setTitle( "Simulation " + zoom + "%" );
	}
	
	protected void zoomIn() {
		simPanel.zoomIn();
		showTitle();
	}
	
	protected void zoomOut() {
		simPanel.zoomOut();
		showTitle();
	}
	
	protected void step() {
		if ( environment != null ) {
			//System.out.println( "step" );
			environment.step();
			repaint();
		}
	}
	
	private SimulationDialog simDialog = null;
	
	protected void newSim() {
		setRunning( false );
		
		if ( simDialog == null )
			simDialog = new SimulationDialog( mainFrame );
		Environment environment = simDialog.createNewSimulation();
		if ( environment != null ) {
			this.environment = environment;
			simPanel.setSimulation( environment.getSimulation() );
		}
	}
	
	protected void savePop() {
		if ( environment == null )
			return;
			
		JFileChooser fileChooser = new JFileChooser();
		if ( fileChooser.showSaveDialog( mainFrame ) == JFileChooser.APPROVE_OPTION ) {
			File file = fileChooser.getSelectedFile();
			try {
				Writer out = new OutputStreamWriter( new BufferedOutputStream( new FileOutputStream( file )));
				DataEncoder encoder = new DataEncoder( out );
				environment.encodePopulationGenotypes( encoder );
				encoder.close();
			}
			catch( IOException ioe ) {
				ioe.printStackTrace();
			}
		}
	}
	
	protected void loadPop() {
		if ( environment == null )
			return;
			
		JFileChooser fileChooser = new JFileChooser();
		if ( fileChooser.showOpenDialog( mainFrame ) == JFileChooser.APPROVE_OPTION ) {
			File file = fileChooser.getSelectedFile();
			loadPopulation( file.getAbsolutePath() );
		}
	}
	
	private void loadPopulation( String fileName ) {
		try {
			Reader in = null;
			if ( !fileName.endsWith( ".gz" ) )
				in = new InputStreamReader( new BufferedInputStream( new FileInputStream( fileName )));
			else
				in = new InputStreamReader( new GZIPInputStream( new BufferedInputStream( new FileInputStream( fileName ))));
		
			DataDecoder decoder = new DataDecoder( in );
			environment.decodePopulationGenotypes( decoder );
			decoder.close();
		}
		catch( IOException ioe ) {
			ioe.printStackTrace();
		}
		simPanel.repaint();
	}
	
	protected void changeRender() {
		render = !render;
		renderMenuItem.setState( render );
	}
	
	protected void viewStats() {
		// TODO viewStats
	}
	
	private String toTime( int time ) {
		int mins = (time/60) % 60;
		int hours = time/3600;
		int secs = time%60;
		
		StringBuffer buffer = new StringBuffer();
		if ( hours < 10 ) buffer.append( '0' );
		buffer.append( hours );
		buffer.append( ':' );
		
		if ( mins < 10 ) buffer.append( '0' );
		buffer.append( mins );
		buffer.append( ':' );
		
		if ( secs < 10 ) buffer.append( '0' );
		buffer.append( secs );
		return buffer.toString();
	}
	
	private MessageFormat speedFormat = new MessageFormat( "  x{0,number,#.##}" );
	
	private void repaint() {
		if ( render && !isMinimised() ) {
			simPanel.repaint();
		}
		
		if ( environment != null ) {
			int timestep = environment.getSimulation().getTimeStep();
			int update = (render && !isMinimised())? 25 : 250;
			if ( timestep % update == 0 ) {
				timeStepLabel.setText( toTime( timestep/25 ) );
				timeStepLabel.repaint();
				speedLabel.setText( speedFormat.format( new Object[]{ new Float( speed ) } ) );
				speedLabel.repaint();
				offspringLabel.setText( environment.getOffspringCount() + " offspring" );
				offspringLabel.repaint();
				fitnessLabel.setText( String.valueOf( environment.getAverageFitness() ) );
				fitnessLabel.repaint();
			}	
		}	
	}
	
	private boolean isMinimised() {
		return (mainFrame.getExtendedState() & Frame.ICONIFIED) != 0;
	}
	
	public void run() {
		long start = System.currentTimeMillis();
		int startTimeSteps = 0;
		
		Thread.currentThread().setPriority( Thread.MIN_PRIORITY );
		while( mainFrame.isVisible() ) {
			
			if ( running && environment != null ) {
				environment.step();
				repaint();
				if ( environment.getSimulation().getTimeStep() % 100 == 0 ) {
					speed = (100/25)*1000/(float)(System.currentTimeMillis()-start);
					start = System.currentTimeMillis();
				}
			}
			else {
				
				try {
					Thread.sleep( 100 );
				}
				catch( Exception e ){}
			}
			
		}
		
	}
	
	private void initMainMenu() {
		// FILE MENU
		fileMenu     = new JMenu( "File" );
		
		newSimulationMenuItem = new JMenuItem( "New Simulation" );
		setAccel( newSimulationMenuItem, KeyEvent.VK_N );
		newSimulationMenuItem.addActionListener( new ActionAdapter( this, "newSim" ) );
		
		savePopulationMenuItem = new JMenuItem( "Save Population" );
		setAccel( savePopulationMenuItem, KeyEvent.VK_S );
		savePopulationMenuItem.addActionListener( new ActionAdapter( this, "savePop" ) );
		
		loadPopulationMenuItem = new JMenuItem( "Load Population" );
		setAccel( loadPopulationMenuItem, KeyEvent.VK_L );
		loadPopulationMenuItem.addActionListener( new ActionAdapter( this, "loadPop" ) );
		
		quitMenuItem = new JMenuItem( "Quit" );
		setAccel( quitMenuItem, KeyEvent.VK_Q );
		quitMenuItem.addActionListener( new ActionListener() {
			public void actionPerformed( ActionEvent ae ) {
				setRunning( false );
				mainFrame.dispatchEvent( new WindowEvent(mainFrame, WindowEvent.WINDOW_CLOSING));
			}
		});
		
		fileMenu.add( newSimulationMenuItem );
		fileMenu.addSeparator();
		fileMenu.add( savePopulationMenuItem );
		fileMenu.add( loadPopulationMenuItem );
		fileMenu.addSeparator();
		fileMenu.add( quitMenuItem );
		////////////////
		
		// CONTROLS MENU
		controlsMenu = new JMenu( "Controls" );
		
		runMenuItem = new JMenuItem( "Run" );
		setAccel( runMenuItem, KeyEvent.VK_R );
		runMenuItem.addActionListener( new ActionListener() {
			public void actionPerformed( ActionEvent ae ) {
				setRunning( true );
			}
		});
		
		pauseMenuItem = new JMenuItem( "Pause" );
		setAccel( pauseMenuItem, KeyEvent.VK_PERIOD );
		pauseMenuItem.addActionListener( new ActionListener() {
			public void actionPerformed( ActionEvent ae ) {
				setRunning( false );
			}
		});
		
		stepMenuItem  = new JMenuItem( "Step" );
		setAccel( stepMenuItem, KeyEvent.VK_COMMA );
		stepMenuItem.addActionListener( new ActionAdapter( this, "step" ) );
		
		controlsMenu.add( runMenuItem );
		controlsMenu.add( pauseMenuItem );
		controlsMenu.add( stepMenuItem );
		////////////////
		
		// VIEW MENU
		viewMenu     = new JMenu( "View" );
		
		zoomInMenuItem  = new JMenuItem( "Zoom In" );
		setAccel( zoomInMenuItem, KeyEvent.VK_I );
		zoomInMenuItem.addActionListener( new ActionAdapter( this, "zoomIn" ) );
		
		zoomOutMenuItem = new JMenuItem( "Zoom Out" );
		setAccel( zoomOutMenuItem, KeyEvent.VK_O );
		zoomOutMenuItem.addActionListener( new ActionAdapter( this, "zoomOut" ) );
		
		renderMenuItem = new JCheckBoxMenuItem( "Render" );
		renderMenuItem.addActionListener( new ActionAdapter( this, "changeRender" ) );
		renderMenuItem.setState( render );
		
		statsMenuItem   = new JMenuItem( "Statistics" );
		statsMenuItem.addActionListener( new ActionAdapter( this, "viewStats" ) );
		
		viewMenu.add( zoomInMenuItem );
		viewMenu.add( zoomOutMenuItem );
		viewMenu.addSeparator();
		viewMenu.add( renderMenuItem );
		viewMenu.addSeparator();
		viewMenu.add( statsMenuItem );
		////////////////
		
		JMenuBar menuBar = new JMenuBar();
		
		menuBar.add( fileMenu );
		menuBar.add( controlsMenu );
		menuBar.add( viewMenu );
		
		mainFrame.setJMenuBar( menuBar );
	}
	
	private void setAccel( JMenuItem item, int keyCode ) {
		item.setAccelerator( 
			KeyStroke.getKeyStroke( keyCode,
				Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() ) 
			);
	}
	
	private void loadEnv( String fileName ) {
		try {
			Parameters params = new Parameters();
			InputStream in = new BufferedInputStream( new FileInputStream( fileName ) );
			params.load( in );
			in.close();
			environment = new Environment( params, new Parameters[ 0 ] );
			simPanel.setSimulation( environment.getSimulation() );
		}
		catch( IOException ioe ) {
			ioe.printStackTrace();
		}
	}
	
	public static void main( String[] args ) {
		
		try {
			// just in case this messes up it's in a try block
			System.setProperty("apple.laf.useScreenMenuBar","true");
		}
		catch( Throwable t ) {}
		
		MainGUI mainGUI = new MainGUI();
		mainGUI.init();
		if ( args.length == 2 ) {
			mainGUI.loadEnv( args[ 0 ] );
			mainGUI.loadPopulation( args[ 1 ] );
		}
		mainGUI.run();
	}
	
}
