/*
 * Created on Nov 23, 2003
 *
 */
package proj1;

import proj1.evol.Genotype;

import proj1.simulator.Color;
import java.io.IOException;
import java.util.Random;


/**
 * @author john
 *
 */
public class AgentGenotype implements Genotype {
	private NeuralNetGenotype nnetGenotype = null;
	private int[] sensorColors = null;
	private float[] sensorSensitivities = null;
	private int numSensorGroups = 2;
	private float sensorAngle = (float)Math.PI;
	private int numHiddenNodes = 0;
	private float motorAngle = (float)(Math.PI/2);
	private float motorPower = 10.0f;
	private float sensorRange = 32;
	private Random rand = new Random();
	private final static float[] initialWeightSet = { -1.5f, -1.0f, 0.5f, 0.0f, 0.5f, 1.0f, 1.5f };
	
	public AgentGenotype( DataDecoder decoder ) throws IOException {
		decode( decoder );
	}
	
	public AgentGenotype( int numSensorGroups, float sensorAngle, int numSensorColors, float sensorRange, int numHiddenNodes, float motorPower ) {
		// always an extra neuron for fixed input
		nnetGenotype = new NeuralNetGenotype( initialWeightSet, 1 + numSensorGroups*(numSensorColors +1), numHiddenNodes, 2 );
		this.numSensorGroups = numSensorGroups;
		this.sensorAngle = sensorAngle;
		this.numHiddenNodes = numHiddenNodes;
		sensorColors = new int[ numSensorColors ];
		sensorSensitivities = new float[ numSensorColors ];
		for ( int i = 0; i < numSensorColors; i++ ) {
			sensorColors[ i ] = 0x00FFFFFF & rand.nextInt();
			sensorSensitivities[ i ] = 0.0f;	
		}
		this.sensorRange = sensorRange;
		this.motorPower = motorPower;
	}
	
	public void encode( DataEncoder encoder ) throws IOException {
		Color[] colors = new Color[ sensorColors.length ];
		for ( int i = 0; i < colors.length; i++ ) colors[ i ] = new Color( sensorColors[ i ] );
		encoder.encode( "AgentGenotype.sensorColors", colors );
		encoder.encode( "AgentGenotype.sensorSensitivities", sensorSensitivities );
		encoder.encode( "AgentGenotype.numSensorGroups", new Integer( numSensorGroups ) );
		encoder.encode( "AgentGenotype.sensorAngle", new Float( sensorAngle ) );
		encoder.encode( "AgentGenotype.sensorRange", new Float( sensorRange ) );
		encoder.encode( "AgentGenotype.numHiddenNodes", new Integer( numHiddenNodes ) );
		encoder.encode( "AgentGenotype.motorPower", new Float( motorPower ) );
		encoder.encode( "AgentGenotype.motorAngle", new Float( motorAngle ) );
		nnetGenotype.encode( encoder );
	}
	
	private void decode( DataDecoder decoder ) throws IOException {
		Color[] colors = decoder.decodeColorArray( "AgentGenotype.sensorColors" );
		sensorColors = new int[ colors.length ];
		for ( int i = 0; i < colors.length; i++ ) sensorColors[ i ] = colors[ i ].getRGB() & 0x00FFFFFF;
		sensorSensitivities = decoder.decodeFloatArray( "AgentGenotype.sensorSensitivities" );
		numSensorGroups = decoder.decodeInt( "AgentGenotype.numSensorGroups" );
		sensorAngle = decoder.decodeFloat( "AgentGenotype.sensorAngle" );
		sensorRange = decoder.decodeFloat( "AgentGenotype.sensorRange" );
		numHiddenNodes = decoder.decodeInt( "AgentGenotype.numHiddenNodes" );
		motorPower = decoder.decodeFloat( "AgentGenotype.motorPower" );
		motorAngle = decoder.decodeFloat( "AgentGenotype.motorAngle" );
		nnetGenotype = new NeuralNetGenotype( decoder );
	}
	
	public int[] getSensorColors() {
		return sensorColors;
	}
	
	public float[] getSensorSensitivities() {
		return sensorSensitivities;
	}
		
	public int getNumSensorGroups() {
		return numSensorGroups;
	}
	
	public float getSensorAngle() {
		return sensorAngle;
	}
	
	public float getSensorRange() {
		return sensorRange;
	}
	
	public int getNumHiddenNodes() {
		return numHiddenNodes;
	}

	public NeuralNetGenotype getNeuralNetGenotype() {
		return nnetGenotype;
	}

	public float getMotorPower() {
		return motorPower;
	}
	
	public float getMotorAngle() {
		return motorAngle;
	}

	/* (non-Javadoc)
	 * @see proj1.evol.Genotype#make()
	 */
	public Genotype make() {
		return new AgentGenotype( numSensorGroups, sensorAngle, sensorColors.length, sensorRange, numHiddenNodes, motorPower );
	}

	/* (non-Javadoc)
	 * @see proj1.evol.Genotype#copy(proj1.evol.Genotype)
	 */
	public void copy(Genotype original) {
		AgentGenotype ag = (AgentGenotype)original;
		nnetGenotype.copy( ag.nnetGenotype );
		System.arraycopy( ag.sensorColors, 0, sensorColors, 0, sensorColors.length );
		System.arraycopy( ag.sensorSensitivities, 0, sensorSensitivities, 0, sensorSensitivities.length );
		numSensorGroups = ag.numSensorGroups;
		sensorAngle = ag.sensorAngle;
		numHiddenNodes = ag.numHiddenNodes;
		motorAngle = ag.motorAngle;
	}

	private int mutateComponent( int c ) {
		return (int)Math.min( 255, Math.max( 0, c + 2*rand.nextGaussian() ) );
	}

	private void mutateSensorColors() {
		int index = (int)(sensorColors.length*Math.random());
		int rgb = sensorColors[ index ];
		int r = (rgb >> 16) & 0xFF;
		int g = (rgb >> 8)  & 0xFF;
		int b = (rgb)       & 0xFF;
		
		if ( Math.random() < 0.3 ) {
			r = mutateComponent( r );
		}
		if ( Math.random() < 0.3 ) {
			g = mutateComponent( g );
		}
		if ( Math.random() < 0.3 ) {
			b = mutateComponent( b );
		}
		
		sensorColors[ index ] = (r << 16 | g << 8 | b);
	}
	
	private void mutateSensorSensitivities() {
		int index = (int)(sensorSensitivities.length*Math.random());
		float sensitivity = sensorSensitivities[ index ];
		sensitivity += 0.05f*rand.nextGaussian();
		sensorSensitivities[ index ] = Math.max( 0.0f, Math.min( 1.0f, sensitivity ) );
	}

	private void mutateSensorAngle() {
		sensorAngle = (float)(sensorAngle + 0.05*Math.PI*rand.nextGaussian());
		sensorAngle = (float)Math.max( 0.0, Math.min( 2*Math.PI, sensorAngle ) );
	}
	
	private void mutateMotorAngle() {
		motorAngle = (float)(motorAngle + 0.05*Math.PI*rand.nextGaussian());
		motorAngle = (float)Math.max( 0.0, Math.min( Math.PI, motorAngle ) );
	}

	/* (non-Javadoc)
	 * @see proj1.evol.Genotype#mutate()
	 */
	public void mutate() {
		do {
			if ( Math.random() < 0.9 )
				nnetGenotype.mutate();
			else {
				if ( Math.random() < 0.9 )
					if ( Math.random() < 0.9 )
						mutateSensorColors();
					else
						mutateSensorSensitivities();
				else if ( Math.random() < 0.5 )
					mutateSensorAngle();
				else
					mutateMotorAngle();
			}
		}
		while( Math.random() < 0.1 );
	}

	/* (non-Javadoc)
	 * @see proj1.evol.Genotype#recombine(proj1.evol.Genotype, proj1.evol.Genotype)
	 */
	public void recombine(Genotype parent1, Genotype parent2) {
		AgentGenotype ag1 = (AgentGenotype)parent1;
		AgentGenotype ag2 = (AgentGenotype)parent2;
		
		for ( int i = 0; i < sensorColors.length; i++ ) {
			double r = Math.random();
			sensorColors[ i ] = r < 0.5? ag1.sensorColors[ i ] : ag2.sensorColors[ i ];
			sensorSensitivities[ i ] = r < 0.5? ag1.sensorSensitivities[ i ] : ag2.sensorSensitivities[ i ];
		}
		
		nnetGenotype.recombine( ag1.nnetGenotype, ag1.nnetGenotype );
		
		float x = (float)(Math.random());
		sensorAngle = x*ag1.sensorAngle + (1-x)*ag2.sensorAngle;
	}

}
