/*
 * Created on Oct 19, 2003
 *
 */
package proj1.simulator;

import proj1.simulator.physics.*;

import java.util.List;
import java.util.ArrayList;


/**
 * @author john
 *
 */
public class Vehicle implements VisibleObject {
	
	private final Body body = new Body();
	private float radius = 5.0f;
	private Motor leftMotor  = null;
	private Motor rightMotor = null;
	
	
	private SensorGroup[] sensorGroups = null;
	private double sensorAngle = Math.PI;
	
	private Brain brain = null;
	private Sensor[] sensors = null;
	private Effector[] effectors = null;
	
	private List nearbyObjects = new ArrayList();
	
	private Color color = Color.blue;
	
	public Vehicle( Brain brain, int numSensorGroups, double sensorAngle, int[] sensorColors, float[] sensorSensitivities, float sensorRange, float motorPower, float motorAngle ) {
		this.brain = brain;
	    body.setMass( 1.0f );
	    body.setMomentOfInertia( radius*radius*body.getMass()*0.5f );
		
		leftMotor  = new Motor(  motorAngle, radius, motorPower );
		rightMotor = new Motor( -motorAngle, radius, motorPower );
		
		this.sensorAngle = sensorAngle;
		sensorGroups = new SensorGroup[ numSensorGroups ];
		float fieldOfView = (float)(sensorAngle/numSensorGroups);
		int numSensors = numSensorGroups*(sensorColors.length+1);
		sensors = new Sensor[ numSensors ];
		
		for ( int i = 0, n = 0; i < numSensorGroups; i++ ) {
			// angle is in middle of field of view
			float angle = (float)(i*sensorAngle/numSensorGroups + (fieldOfView - sensorAngle)/2);
			SensorGroup group = new SensorGroup( this, angle, fieldOfView );
			sensorGroups[ i ] = group;
			Sensor light = new LightSensor( sensorRange );
			group.add( light );
			sensors[ n ] = light;
			n++;
			for ( int j = 0; j < sensorColors.length; j++, n++ ) {
				Sensor color = new ColorSensor( new Color( sensorColors[ j ] ), sensorRange );
				color.setSensitivity( sensorSensitivities[ j ] );
				group.add( color );
				sensors[ n ] = color;
			}
		}
		
		effectors = new Motor[] { leftMotor, rightMotor	};
	}
        
	public Color getColor() {
		return color;
	}
	
	public void setColor( Color color ) {	
		this.color = color;
	}
	
	/**
	 * @return
	 */
	public float getRadius() {
		return radius;
	}

	/**
	 * @param f
	 */
	public void setRadius(float f) {
		radius = f;
	}
        
	public float getAngle() {
	    return body.getAngle();
	}
	
	public Vector2d getPosition() {
	    return body.getPosition();
	}
	
	public void setAngle( float angle ) {
		body.setAngle( angle );
	}
	
	public Body getBody() {
	    return body;
	}
	
	public double getSensorAngle() {
		return sensorAngle;
	}
	
	public int getNumSensorGroups() {
		return sensorGroups.length;
	}
	
	public SensorGroup[] getSensorGroups() {
		return sensorGroups;
	}
	
	public float getMaxSensorRange() {
		float max = 0.0f;
		for ( int i = 0; i < sensorGroups.length; i++ )
			max = Math.max( max, sensorGroups[ i ].getMaxRange() );
		return max;
	}
	
	int count = 0;
	
	public void applyForces() {
		
		if ( brain != null ) {
			brain.receiveInputs( sensors );
			brain.computeOutputs( effectors );
		}
		
		leftMotor.apply();
		rightMotor.apply();
	}

	public void integrate( float dt ) {
	    body.integrate( dt );
	}
	
	public void update() {
	    body.update();
	}
	
	public void updateSensors( Vehicle vehicle ) {
		if ( !nearbyObjects.contains( vehicle ) )
			nearbyObjects.add( vehicle );
	}
	
	public void updateSensors( FixedObject obj ) {
		if ( !nearbyObjects.contains( obj ) )
			nearbyObjects.add( obj );
	}
	
	public void updateSensors( int width, int height ) {
		//if ( nearbyVehicles.size() != 0 )
		//	System.out.println( this );
		for ( int i = 0; i < sensorGroups.length; i++ )
			sensorGroups[ i ].updateSensors( nearbyObjects, width, height );
		
		//if ( nearbyVehicles.size() != 0 )
		//	System.out.println();
		
		nearbyObjects.clear();
		
	}
	
	public Motor getMotor( int i ) {
	    switch( i ) {
	        case 0: return leftMotor;
	        case 1: return rightMotor;
	    }
	    return null;
	}
	
	/**
	 * @author john
	 *
	 */
	public class Motor implements Effector {
		private float output = 0.0f;
		private float power = 20.0f;
		private float angle = 0.0f;
		private float offset = 0.0f;
		private Vector2d v = new Vector2d();
		
		public Motor( float angle, float offset, float power ) {
			this.angle  = angle;
			this.offset = offset;
			this.power = power;
		}
		
		public void setOutput( float output ) {
			this.output = output;
		}
		
		public Vector2d getPosition( Vector2d v ) {
			float angle = body.getAngle() + this.angle;
			v.setPolar( angle, offset );
			v.add( body.getPosition() );
			return v;
		}
		
		public void apply() {
			getPosition( v );
			
			float x = v.x, y = v.y;
			
			float angle = body.getAngle();
			v.setPolar( angle, output*power );
			
			float fx = v.x, fy = v.y;
			
			body.applyForce( x, y, fx, fy );
		}
		
	}
	

}
