/*
 * Created on Nov 12, 2003
 *
 */
package proj1;

import java.io.IOException;
import java.util.Random;

import proj1.evol.Genotype;

import proj1.nnet.*;

/**
 * @author john
 *
 */
public class NeuralNetGenotype implements Genotype {
	private static Random rand = new Random();
	private int numInputs, numOutputs;
	private int numHidden = 0;
	private float[] weights = null;
	private float[] biases = null;
	
	public NeuralNetGenotype( float[] initialWeightSet, int numInputs, int numHidden, int numOutputs ) {
		this.numInputs = numInputs;
		this.numHidden = numHidden;
		this.numOutputs = numOutputs;
		
		biases = new float[ numHidden + numInputs + numOutputs ];
		for ( int i = 0; i < biases.length; i++ ) biases[ i ] = 0.0f;
		
		int numWeights = numHidden == 0? (numInputs*numOutputs) : (numInputs*numHidden + numOutputs*numHidden);
		weights = new float[ numWeights ];
		
		for ( int i = 0; i < numWeights; i++ ) {
			if ( initialWeightSet != null )
				weights[ i ] = initialWeightSet[ (int)(initialWeightSet.length*rand.nextDouble()) ];
			else
				weights[ i ] = (float)rand.nextGaussian();
		}
	}
	
	public NeuralNetGenotype( int numInputs, int numHidden, int numOutputs ) {
		this( null, numInputs, numHidden, numOutputs );
	}

	public NeuralNetGenotype( DataDecoder decoder ) throws IOException {
		decode( decoder );
	}

	public void encode( DataEncoder encoder ) throws IOException {
		encoder.encode( "NeuralNetGenotype.numInputs", new Integer( numInputs ) );
		encoder.encode( "NeuralNetGenotype.numOutputs", new Integer( numOutputs ) );
		encoder.encode( "NeuralNetGenotype.numHidden", new Integer( numHidden ) );
		encoder.encode( "NeuralNetGenotype.weights", weights );
		encoder.encode( "NeuralNetGenotype.biases", biases );
	}
	
	public void decode( DataDecoder decoder ) throws IOException {
		numInputs = decoder.decodeInt( "NeuralNetGenotype.numInputs" );
		numOutputs = decoder.decodeInt( "NeuralNetGenotype.numOutputs" );
		numHidden = decoder.decodeInt( "NeuralNetGenotype.numHidden" );
		weights = decoder.decodeFloatArray( "NeuralNetGenotype.weights" );
		biases = decoder.decodeFloatArray( "NeuralNetGenotype.biases" );
	}

	public void initNeuralNet( NeuralNet nnet ) {
		// TODO enforce symmetry?
		int ni = 0;
		for ( int i = 0; i < numInputs; i++, ni++ ) {
			Neuron n = new Neuron( biases[ ni ] );
			nnet.add( n );
			nnet.addInput( n );
		}
		
		for ( int i = 0; i < numHidden; i++, ni++ ) {
			Neuron n = new Neuron( biases[ ni ] );
			nnet.add( n );
		}
		
		for ( int i = 0; i < numOutputs; i++, ni++ ) {
			Neuron n = new Neuron( biases[ ni ] );
			nnet.add( n );
			nnet.addOutput( n );
		}
		
		if ( numHidden == 0 ) {
			// direct connection (no hidden nodes)
			for ( int i = 0, n = 0; i < numInputs; i++ ) {
				for ( int j = 0; j < numOutputs; j++, n++ ) {
					nnet.add( nnet.getInput( i ), weights[ n ], nnet.getOutput( j ) );
				}
			}
		}
		else {
			// everything via hidden nodes
			int n = 0;
			// input to hidden
			for ( int i = 0; i < numInputs; i++ ) {
				for ( int j = 0; j < numHidden; j++, n++ ) {
					nnet.add( nnet.getInput( i ), weights[ n ], nnet.get( numInputs+j ) );
				}
			}
			
			// hidden to output
			for ( int i = 0; i < numHidden; i++ ) {
				for ( int j = 0; j < numOutputs; j++, n++ ) {
					nnet.add( nnet.get( numInputs+i ), weights[ n ], nnet.getOutput( j ) );
				}
			}
		}
	}

	public float manhattenDistance( NeuralNetGenotype nngeno ) {
		float sum = 0.0f;
		for ( int i = 0; i < biases.length; i++ ) sum += Math.abs( biases[ i ] - nngeno.biases[ i ] );
		for ( int i = 0; i < weights.length; i++ ) sum += Math.abs( weights[ i ] - nngeno.weights[ i ] );
		return sum;
	}

	/* (non-Javadoc)
	 * @see proj1.evol.Genotype#make()
	 */
	public Genotype make() {
		return new NeuralNetGenotype( numInputs, numHidden, numOutputs );
	}

	/* (non-Javadoc)
	 * @see proj1.evol.Genotype#copy(proj1.evol.Genotype)
	 */
	public void copy(Genotype original) {
		NeuralNetGenotype nngeno = (NeuralNetGenotype)original;
		System.arraycopy( nngeno.weights, 0, weights, 0, weights.length );
		System.arraycopy( nngeno.biases, 0, biases, 0, biases.length );
		numInputs = nngeno.numInputs;
		numOutputs = nngeno.numOutputs;
		numHidden = nngeno.numHidden;
	}

	private void mutate( float[] values ) {
		int index = (int)(Math.random()*values.length);
		if ( Math.random() < 0.9 )
			values[ index ] += (float)(0.05f*(rand.nextGaussian()));
		else {
			// occasionally reverse a weight or duplicate it
			if ( Math.random() < 0.25f )
				values[ index ] = -values[ index ];
			else
				values[ (int)(Math.random()*values.length) ] = values[ index ];
	
		}
	}

	/* (non-Javadoc)
	 * @see proj1.evol.Genotype#mutate()
	 */
	public void mutate() {
		if ( Math.random() < 0.75f )
			mutate( weights );
		else
			mutate( biases );
	}

	/* (non-Javadoc)
	 * @see proj1.evol.Genotype#recombine(proj1.evol.Genotype, proj1.evol.Genotype)
	 */
	public void recombine(Genotype parent1, Genotype parent2) {
		NeuralNetGenotype nn1 = (NeuralNetGenotype)parent1;
		NeuralNetGenotype nn2 = (NeuralNetGenotype)parent2;
		//for ( int i = 0; i < weights.length; i++ )
		//	weights[ i ] = Math.random() < 0.5? nn1.weights[ i ] : nn2.weights[ i ];
		
		// attempt to swap "functional block" of mapping from input to output	
		for ( int i = 0, n = 0; i < numInputs; i++ ) {
			boolean boolP1 = Math.random() < 0.5f;
			for ( int j = 0; j < numOutputs; j++, n++ ) {
				weights[ n ] = boolP1? nn1.weights[ n ] : nn2.weights[ n ];
			}
		}
	}
	
	public String toString() {
		StringBuffer buffer = new StringBuffer( getClass().getName() + "-weights[" );
		for ( int i = 0; i < weights.length; i++ ) {
			if ( i != 0 ) buffer.append( ", " );
			buffer.append( weights[ i ] );
		}
		buffer.append( "]" );
		return buffer.toString();
	}

	

}
