package proj1;
/*
 * Created on Nov 4, 2003
 *
 */

import proj1.simulator.*;
import proj1.simulator.physics.Vector2d;
import proj1.evol.*;

import java.io.*;
import java.util.*;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

/**
 * @author john
 *
 */
public class Environment implements CollisionHandler {
	
	private Population[] populations = null;
	private List food = new ArrayList();
	private int foodLevel = 100;
	private Simulation sim = null;
	private int birthRate = 100;
	private int startupTime = 900;
	private int offspringCount = 0;
	private Color[] populationColors = { Color.blue, Color.red, Color.yellow, Color.cyan, Color.magenta, Color.pink,Color.orange };
	private String dataDirectory = "";
	private Writer dataWriter = null;
	private float penalty = 0.2f;
	private float reward = 20.0f;
	
	private int herbivoreFoodCollisions = 0;
	private int carnivoreHerbivoreCollisions = 0;
	private int otherCollisions = 0;
	
	public Environment( Parameters params, Parameters[] speciesParameters ) {
		initFromParams( params, speciesParameters );
	}
	
	private void initFromParams( Parameters params, Parameters[] speciesParameters ) {
		int width  = params.getParameter( "width", 1500 );
		int height = params.getParameter( "height", 1500 );
		int rocks  = params.getParameter( "rocks", 100 );
		sim = new Simulation(width, height );
		sim.setCollisionHandler( this );
		
		for ( int i = 0; i < rocks; i++ ) {
			Rock rock = new Rock( (float)(Math.random()*width), (float)(Math.random()*height) );
			sim.addFixedObject( rock );
		}
		
		foodLevel = params.getParameter( "foodLevel", 100 );
		birthRate = params.getParameter( "birthRate", 100 );
		penalty = (float)params.getParameter( "penalty", 0.2 );
		reward = (float)params.getParameter( "reward", 20.0 );
		
		dataDirectory = params.getParameter( "dataDirectory", "data" );
		
		int numPopulations = speciesParameters.length;
		populations = new Population[ numPopulations ];
		
		for ( int i = 0; i < numPopulations; i++ ) {
			Parameters species = speciesParameters[ i ];
			
			String name = species.getParameter( "name", "species" + i );
			double mutationRate = species.getParameter( "mutationRate", 0.5 );
			double recombinationRate = species.getParameter( "recombinationRate", 0.5 );
			populations[ i ] = new Population( name, mutationRate, recombinationRate );
			
			Color defaultColor = populationColors[ i % populationColors.length ];
			// defaults to blue
			Color color = new Color( species.getParameter( "color", defaultColor.getRGB() ) );
			int popSize = species.getParameter( "populationSize", 50 );
			int numSensorGroups = species.getParameter( "numSensorGroups", 2 );
			double sensorAngle = species.getParameter( "sensorAngle", Math.PI );
			int numHidden = species.getParameter( "numHiddenNodes", 0 );
			int numSensorColors = species.getParameter( "numSensorColors", 1 );
			float sensorRange = (float)species.getParameter( "sensorRange", 64.0 );
			float motorPower = (float)species.getParameter( "motorPower", 10.0 );
			boolean carnivore = species.getParameter( "carnivore", false );
			boolean herbivore = species.getParameter( "herbivore", true );
			for ( int j = 0; j < popSize; j++ ) {
				Agent agent = new Agent( new AgentGenotype( numSensorGroups, (float)sensorAngle, numSensorColors, sensorRange, numHidden, motorPower ), carnivore, herbivore );
				agent.setColor( color );
				placeRandomly( agent );
				populations[ i ].add( agent );
				sim.addVehicle( agent );
			}
		}
		
		try {
			File dir  = new File( dataDirectory );
			// TODO check if dir already exists and warn user?
			dir.mkdirs();

			File file = new File( dataDirectory, "params.env" );
			OutputStream out = new BufferedOutputStream( new FileOutputStream( file ) );
			params.save( out );
			out.flush();
			out.close();

			for ( int i = 0; i < numPopulations; i++ ) {
				Parameters species = speciesParameters[ i ];
				String name = species.getParameter( "name", "species"+i );
				file = new File( dataDirectory, name + ".species" );
				out = new BufferedOutputStream( new FileOutputStream( file ) );
				species.save( out );
				out.flush();
				out.close();
			}

			prepareDataFiles();

		}
		catch( Exception e ){
			throw new RuntimeException( e );
		}
	}
	
	public void prepareDataFiles() {
		try {
			// prepare writer for data
			File dataFile = new File( dataDirectory, "data.csv" );
			OutputStream out = new BufferedOutputStream( new FileOutputStream( dataFile ) );
			dataWriter = new OutputStreamWriter( out );
			writeDataHeader( dataWriter );
		}
		catch( Exception e ){
			throw new RuntimeException( e );
		}
	}
	
	private void writeDataHeader( Writer w ) throws IOException {
		// TODO add sensors colors and sensor angles
		StringBuffer buffer = new StringBuffer();
		buffer.append( "time" );
		buffer.append( ',' );
		buffer.append( "herbivore-food-collisions" );
		buffer.append( ',' );
		buffer.append( "carnivore-herbivore-collisions" );
		buffer.append( ',' );
		buffer.append( "other-collisions" );
		
		for ( int i = 0; i < populations.length; i++ ) {
			Population population = populations[ i ];
			
			String name = population.getName();
			buffer.append( ',' );
			buffer.append( name + "-best-fitness" );
			buffer.append( ',' );
			buffer.append( name + "-worst-fitness" );
			buffer.append( ',' );
			buffer.append( name + "-average-fitness" );
			buffer.append( ',' );
			buffer.append( name + "-average-age" );
			buffer.append( ',' );
			buffer.append( name + "-average-network-variation" );
			buffer.append( ',' );
			buffer.append( name + "-average-motor-angle" );
			buffer.append( ',' );
			buffer.append( name + "-average-motor-angle-variation" );
			buffer.append( ',' );
			buffer.append( name + "-average-sensor-angle" );
			buffer.append( ',' );
			buffer.append( name + "-average-sensor-variation" );
			
			if ( population.size() > 0 ) {
				AgentGenotype ageno = (AgentGenotype)population.get( 0 ).getGenotype();
				int[] sensorColors = ageno.getSensorColors();
				for ( int j = 0; j < sensorColors.length; j++ ) {
					buffer.append( ',' );
					buffer.append( name + "-average-sensor-red-" );
					buffer.append( j );
					buffer.append( ',' );
					buffer.append( name + "-average-sensor-green-" );
					buffer.append( j );
					buffer.append( ',' );
					buffer.append( name + "-average-sensor-blue-" );
					buffer.append( j );
					buffer.append( ',' );
					buffer.append( name + "-average-sensor-sensitivity-" );
					buffer.append( j );
				}
			}
		}
		buffer.append( '\n' );
		w.write( buffer.toString() );
		w.flush();
	}
	
	private void savePopulation() {
		try {
			File file = new File( dataDirectory, "population" + sim.getTimeStep() + ".pop.gz" );
			System.out.print( "Saving population to: " + file.getAbsolutePath() + " ..." );
			System.out.flush();
			Writer out = new OutputStreamWriter( new GZIPOutputStream( new BufferedOutputStream( new FileOutputStream( file ) ) ) );
			DataEncoder encoder = new DataEncoder( out );
			encodePopulationGenotypes( encoder );
			encoder.close();
			System.out.println( "done" );
		}
		catch( IOException ioe ) {
			ioe.printStackTrace();
		}
	}
	
	public void simulationFinished() {
		try {
			savePopulation();
			dataWriter.close();
		}
		catch( IOException ioe ) {
			ioe.printStackTrace();
		}
	}
	
	private float averageNetworkVariation( Population population ) {
		float sum = 0.0f;
		int count = 0;
		for ( int i = 0; i < population.size(); i++ ) {
			AgentGenotype agenoi = (AgentGenotype)population.get( i ).getGenotype();
			for ( int j = i+1; j < population.size(); j++, count++ ) {
				AgentGenotype agenoj = (AgentGenotype)population.get( j ).getGenotype();
				sum += agenoi.getNeuralNetGenotype().manhattenDistance( agenoj.getNeuralNetGenotype() );
			}
		}
		return count != 0? sum/count : 0;
	}
	
	private float averageMotorAngle( Population population ) {
		float sum = 0.0f;
		for ( int i = 0; i < population.size(); i++ ) {
			AgentGenotype ageno = (AgentGenotype)population.get( i ).getGenotype();
			sum += ageno.getMotorAngle();
		}
		return sum/population.size();
	}
	
	private float averageMotorAngleVariation( Population population ) {
		float sum = 0.0f;
		int count = 0;
		for ( int i = 0; i < population.size(); i++ ) {
			AgentGenotype agenoi = (AgentGenotype)population.get( i ).getGenotype();
			for ( int j = i+1; j < population.size(); j++, count++ ) {
				AgentGenotype agenoj = (AgentGenotype)population.get( j ).getGenotype();
				sum += Math.abs( agenoi.getMotorAngle() -  agenoj.getMotorAngle() );
			}
		}
		return count != 0? sum/count : 0;
	}
	
	private float averageSensorAngle( Population population ) {
		float sum = 0.0f;
		for ( int i = 0; i < population.size(); i++ ) {
			AgentGenotype ageno = (AgentGenotype)population.get( i ).getGenotype();
			sum += ageno.getSensorAngle();
		}
		return sum/population.size();
	}
	
	private float averageSensorAngleVariation( Population population ) {
		float sum = 0.0f;
		int count = 0;
		for ( int i = 0; i < population.size(); i++ ) {
			AgentGenotype agenoi = (AgentGenotype)population.get( i ).getGenotype();
			for ( int j = i+1; j < population.size(); j++, count++ ) {
				AgentGenotype agenoj = (AgentGenotype)population.get( j ).getGenotype();
				sum += Math.abs( agenoi.getSensorAngle() -  agenoj.getSensorAngle() );
			}
		}
		return count != 0? sum/count : 0;
	}
	
	private int[] averageSensorColors( Population population ) {
		if ( population.size() <= 0 )
			return new int[ 0 ];
		
		int numSensors = ((AgentGenotype)population.get( 0 ).getGenotype()).getSensorColors().length;
		
		int[][] sums = new int[ numSensors ][ 3 ];
		
		for ( int i = 0; i < population.size(); i++ ) {
			AgentGenotype ageno = (AgentGenotype)population.get( i ).getGenotype();
			int[] sensorColors = ageno.getSensorColors();
			for ( int j = 0; j < numSensors; j++ ) {
				int rgb = sensorColors[ j ];
				int r = (rgb >> 16) & 0xFF;
				int g = (rgb >> 8) & 0xFF;
				int b = (rgb) & 0xFF;
				int[] comSum = sums[ j ];
				comSum[ 0 ] += r;
				comSum[ 1 ] += g;
				comSum[ 2 ] += b;
			}
		}
		
		int[] averages = new int[ numSensors ];
		for ( int i = 0; i < numSensors; i++ ) {
			int[] comSum = sums[ i ];
			int r = comSum[ 0 ]/population.size();
			int g = comSum[ 1 ]/population.size();
			int b = comSum[ 2 ]/population.size();	
			int rgb = (r << 16) | (g << 8) | b;
			averages[ i ] = rgb;
		}
		return averages;
	}
	
	private float[] averageSensorSensitivity( Population population ) {
		if ( population.size() <= 0 )
			return new float[ 0 ];
	
		int numSensors = ((AgentGenotype)population.get( 0 ).getGenotype()).getSensorColors().length;
	
		float[] sums = new float[ numSensors ];
	
		for ( int i = 0; i < population.size(); i++ ) {
			AgentGenotype ageno = (AgentGenotype)population.get( i ).getGenotype();
			float[] sensorSensitivities = ageno.getSensorSensitivities();
			for ( int j = 0; j < numSensors; j++ ) {
				sums[ j ] += sensorSensitivities[ j ];
			}
		}
	
		for ( int i = 0; i < numSensors; i++ ) {
			sums[ i ] = sums[ i ]/population.size();
		}
		return sums;
	}
	
	private void writeData( Writer w ) throws IOException {
		
		StringBuffer buffer = new StringBuffer();
		buffer.append( sim.getTimeStep() );
		
		buffer.append( ',' );
		buffer.append( herbivoreFoodCollisions );
		buffer.append( ',' );
		buffer.append( carnivoreHerbivoreCollisions );
		buffer.append( ',' );
		buffer.append( otherCollisions );
		
		// reset
		herbivoreFoodCollisions = carnivoreHerbivoreCollisions = otherCollisions = 0;
		
		for ( int i = 0; i < populations.length; i++ ) {
			Population population = populations[ i ];
			String name = population.getName();
			buffer.append( ',' );
			buffer.append( population.getBestFitness() );
			buffer.append( ',' );
			buffer.append( population.getWorstFitness() );
			buffer.append( ',' );
			buffer.append( population.calculateAverageFitness() );
			buffer.append( ',' );
			buffer.append( population.calculateAverageAge() );
			buffer.append( ',' );
			buffer.append( averageNetworkVariation( population ) );
			buffer.append( ',' );
			buffer.append( averageMotorAngle( population ) );
			buffer.append( ',' );
			buffer.append( averageMotorAngleVariation( population ) );
			buffer.append( ',' );
			buffer.append( averageSensorAngle( population ) );
			buffer.append( ',' );
			buffer.append( averageSensorAngleVariation( population ) );
			int[] sensorColors = averageSensorColors( population );
			float[] sensorSensitivities = averageSensorSensitivity( population );
			for ( int j = 0; j < sensorColors.length; j++ ) {
				int rgb = sensorColors[ j ] & 0x00FFFFFF;
				int r = (rgb >> 16) & 0xFF;
				int g = (rgb >> 8) & 0xFF;
				int b = (rgb) & 0xFF;
				buffer.append( ',' );
				buffer.append( r );
				buffer.append( ',' );
				buffer.append( g );
				buffer.append( ',' );
				buffer.append( b );
				buffer.append( ',' );
				buffer.append( sensorSensitivities[ j ] );
			}
		}
		buffer.append( '\n' );
		w.write( buffer.toString() );
		w.flush();
	}
	
	public static void encodePopulationGenotypes( Population population, DataEncoder encoder ) throws IOException {
		String name = population.getName();
		encoder.encode( "Population.name", name );
		double mutationRate = population.getMutationRate();
		encoder.encode( "Population.mutationRate", new Double( mutationRate ) );
		double recombinationRate = population.getRecombinationRate();
		encoder.encode( "Population.recombinationRate", new Double( recombinationRate ) );
		int size = population.size();
		encoder.encode( "Population.size", new Integer( size ) );
		encoder.space();
		
		for ( int i = 0; i < size; i++ ) {
			Phenotype phenotype = population.get( i );
			Agent agent = (Agent)phenotype;
			AgentGenotype genotype = (AgentGenotype)phenotype.getGenotype();
			encoder.encode( "Agent.color", agent.getColor() );
			encoder.encode( "Agent.isHerbivore", agent.isHerbivore()? Boolean.TRUE : Boolean.FALSE );
			encoder.encode( "Agent.isCarnivore", agent.isCarnivore()? Boolean.TRUE : Boolean.FALSE );
			genotype.encode( encoder );
			encoder.space();
		}
		encoder.space();
	}
	
	public void encodePopulationGenotypes( DataEncoder encoder ) throws IOException {
		encoder.encode( "Environment.numPopulations", new Integer( populations.length ) );
		encoder.space();
		for ( int i = 0; i < populations.length; i++ )
			encodePopulationGenotypes( populations[ i ], encoder );
	}
	
	public static Population decodePopulationGenotype( DataDecoder decoder ) throws IOException {
		String name = decoder.decodeString( "Population.name" );
		double mutationRate = decoder.decodeDouble( "Population.mutationRate" );
		double recombinationRate = decoder.decodeDouble( "Population.recombinationRate" );
		
		Population population = new Population( name, mutationRate, recombinationRate );
		
		int size = decoder.decodeInt( "Population.size" );
		
		for ( int i = 0; i < size; i++ ) {
			
			Color color = decoder.decodeColor( "Agent.color" );
			boolean herbivore = decoder.decodeBoolean( "Agent.isHerbivore" );
			boolean carnivore = decoder.decodeBoolean( "Agent.isCarnivore" );
			AgentGenotype genotype = new AgentGenotype( decoder );
			Agent agent = new Agent( genotype, carnivore, herbivore );
			
			agent.setColor( color );
			population.add( agent );
		}
		
		return population;
	}
	
	public void decodePopulationGenotypes( DataDecoder decoder ) throws IOException {
		int numPopulations = decoder.decodeInt( "Environment.numPopulations" );
		
		populations = new Population[ numPopulations ];
		
		for ( int i = 0; i < populations.length; i++ )
			populations[ i ] = decodePopulationGenotype( decoder );
			
		sim.removeAllVehicles();
		
		for ( int i = 0; i < populations.length; i++ ) {
			Population population = populations[ i ];
			for ( int j = 0; j < population.size(); j++ ) {
				Agent agent = (Agent)population.get( j );	
				placeRandomly( agent );
				sim.addVehicle( agent );
			}
		}

		prepareDataFiles();
	}
	
	private void placeRandomly( Agent agent ) {
		int width = sim.getWidth();
		int height = sim.getHeight();
		
		agent.getPosition().set( (int)(Math.random()*width), (int)(Math.random()*height) );
	}
	
	public Simulation getSimulation() {
		return sim;
	}
	
	public float getAverageFitness() {
		float sum = 0.0f;
		for ( int i = 0; i < populations.length; i++ )
			sum += populations[ i ].calculateAverageFitness();
		return sum/populations.length;
	}
	
	public int getAverageAge() {
		int sum = 0;
		for ( int i = 0; i < populations.length; i++ )
			sum += populations[ i ].calculateAverageAge();
		return sum/populations.length;
	}
	
	private void makeOffspring() {
		for ( int i = 0; i < populations.length; i++ ) {
			Population population = populations[ i ];
			if ( population.size() == 0 )
				population.promoteJuvenilles();
			else {
				Agent agent = (Agent)population.makeChild();
				Agent dead = (Agent)(Math.random() < 0.9? population.selectWorst() : population.selectRandom());
				population.remove( dead );
				sim.removeVehicle( dead );
				// previous juvenilles can breed next time
				population.promoteJuvenilles();
				placeRandomly( agent );
				population.addJuvenille( agent );
				sim.addVehicle( agent );
				offspringCount++;
			}
		}
	}
	
	public int getOffspringCount() {
		return offspringCount;
	}
	
	private int logTime = 0;
	
	private void logData() {
		
		if ( logTime % 10 == 0 ) {
			try {
				writeData( dataWriter );
			}
			catch( IOException ioe ) {
				throw new RuntimeException( ioe );
			}
		}
		
		if ( logTime % 1000 == 0 ) {
			savePopulation();
		}
		
		logTime++;
		
	}
	
	public void step() {
		sim.step();
	
		if ( sim.getTimeStep() > startupTime && (sim.getTimeStep()) % birthRate == 0 ) {
			// store data before creatng offspring, so as to get more "accurate" fitness
			logData();
			makeOffspring();
		}
		
		// TODO deal with populations being really bad
		
		
		
		if ( food.size() < foodLevel ) {
			int width = sim.getWidth();
			int height = sim.getHeight();
			Food foodSource = new Food( (float)(width*Math.random()), (float)(height*Math.random()));
			food.add( foodSource );
			sim.addFixedObject( foodSource );
		}
	}
	
	
	
	public void handleCollision( Simulation sim, Vehicle veh1, Vehicle veh2 ) {
		
		Agent agent1 = (Agent)veh1;
		Agent agent2 = (Agent)veh2;
		
		// remove at least "reward" worth of energy or a quarter of the agents energy in one go
		if ( agent1.isCarnivore() && !agent2.isCarnivore() ) {
			float energy = Math.max( reward, agent2.getEnergy()*0.25f );
			agent1.changeEnergy( energy );
			agent2.changeEnergy( -energy );
			placeRandomly( agent2 );
			carnivoreHerbivoreCollisions++;
		}
		else if ( agent2.isCarnivore() && !agent1.isCarnivore() ) {
			float energy = Math.max( reward, agent1.getEnergy()*0.25f );
			agent2.changeEnergy( energy );
			agent1.changeEnergy( -energy );
			placeRandomly( agent1 );
			carnivoreHerbivoreCollisions++;
		}
		else {
			// default to standard physical collision
			sim.handleCollision( sim, veh1, veh2 );
			agent1.changeEnergy( -penalty );
			agent2.changeEnergy( -penalty );
			otherCollisions++;
		}
	}
	
	public void handleBoundaryCollision( Simulation sim, Vehicle veh, Vector2d pos, Vector2d normal ) {
//		default to standard physical collision
		sim.handleBoundaryCollision( sim, veh, pos, normal );
		((Agent)veh).changeEnergy( -2*penalty );
		otherCollisions++;
	}
	
	public void handleCollision( Simulation sim, Vehicle veh, FixedObject obj ) {
		Agent agent = (Agent)veh;
		// TODO don't use instanceof (id's maybe?)
		if ( agent.isHerbivore() && obj instanceof Food ) {
			
			food.remove( (Food)obj );
			sim.removeFixedObject( obj );
			((Agent)veh).changeEnergy( reward );
			herbivoreFoodCollisions++;
		}
		else {
			((Agent)veh).changeEnergy( -penalty );
//			default to standard physical collision
			sim.handleCollision( sim, veh, obj );
			otherCollisions++;
		}
	}
	
	private static Map extractArgs( String[] args ) {
		HashMap argProps = new HashMap();
		List populationList = new ArrayList();
		argProps.put( "populations", populationList );
		for ( int i = 0; i < args.length; i++ ) {
			String arg = args[ i ];
			if ( arg.startsWith( "-") ) {
				String argSwitch = arg.substring( 1 );
				if ( argSwitch.equals( "e" ) ) {
					// environment
					i++;
					String environment = args[ i ];
					argProps.put( "environment", environment );
				}
				else if ( argSwitch.equals( "p" ) ) {
					// population
					i++;
					String population = args[ i ];
					argProps.put( "population-file", population );
				}
				else if ( argSwitch.equals( "t" ) ) {
					// time (how long to run the sim for)
					i++;
					String time = args[ i ];
					argProps.put( "time", new Integer( time ) );
				}
				else if ( argSwitch.equals( "d" ) ) {
					// time (how long to run the sim for)
					i++;
					String dir = args[ i ];
					argProps.put( "directory", dir );
				}
				else
					throw new RuntimeException( "Unknown switch: " + argSwitch );
			}
			else {
				populationList.add( arg );
			}
		}
		return argProps;
	}
	
	private static void printUsage() {
		System.out.println( "usage: java " + Environment.class.getName() + "-e <environment-file> -p <population-file> -t <sim-duration> -d <directory> <species-params>*" );
	}
	
	public static void main( String[] args ) throws IOException {
		Map argsMap = null;
		try {
			argsMap = extractArgs( args );
		}
		catch( Exception e ) {
			e.printStackTrace();
			printUsage();
			System.exit( -1 );
		}
		
		String envFile = (String)argsMap.get( "environment" );
		String popFile = (String)argsMap.get( "population-file" );
		String dir     = (String)argsMap.get( "directory" );
		Integer time   = (Integer)argsMap.get( "time" );
		List populations = (List)argsMap.get( "populations" );
		
		Parameters envParams = new Parameters();
		if ( envFile != null ) {
			System.out.println( "Reading environment params from: " + envFile );
			InputStream in = new BufferedInputStream( new FileInputStream( envFile ) );
			envParams.load( in );
			in.close();
		}
		else {
			System.out.println( "Using defaults for environment params" );
		}
		
		if ( dir != null )
			envParams.setParameter( "dataDirectory", dir );
			
		Parameters[] popParams = new Parameters[ populations.size() ];
		for ( int i = 0; i < popParams.length; i++ ) {
			String popParam = (String)populations.get( i );
			System.out.println( "Reading population params from: " + popParam );
			popParams[ i ] = new Parameters();
			InputStream in = new BufferedInputStream( new FileInputStream( popParam ) );
			popParams[ i ].load( in );
			in.close();
		}
		
		Environment env = new Environment( envParams, popParams );
		
		if ( popFile != null ) {
			System.out.println( "Loading population from: " + popFile );
			Reader in = null;
			if ( !popFile.endsWith( ".gz" ) )
				in = new InputStreamReader( new BufferedInputStream( new FileInputStream( popFile ) ) );
			else
				in = new InputStreamReader( new GZIPInputStream( new BufferedInputStream( new FileInputStream( popFile ) ) ) );
			
			DataDecoder decoder = new DataDecoder( in );
			env.decodePopulationGenotypes( decoder );
			decoder.close();
		}
		
		System.out.println();
		
		System.out.println( env.populations.length + " populations" );
		for ( int i = 0; i < env.populations.length; i++ ) {
			Population pop = env.populations[ i ];
			System.out.println( pop.size() + " \'" + pop.getName() + "\'" );
		}
		
		System.out.println();
		
		File file = new File( envParams.getParameter( "dataDirectory", "data" ) );
		
		System.out.println( "Output data will be stored in: " + file.getAbsolutePath() );
		
		System.out.println();
		
		int numSteps = Integer.MAX_VALUE;
		if ( time != null )
			numSteps = time.intValue();
			
		System.out.println( "Running Simulation for " + numSteps + " timesteps (about " + (numSteps/(25*60*60)) + " hours)" );
		
		long start = System.currentTimeMillis();
		for ( int i = 0; i < numSteps; i++ ) {
			env.step();
			if ( i % 10000 == 0 && i != 0 ) {
				long duration = System.currentTimeMillis()-start;
				double speed = i/(double)duration;
				//System.out.println( speed );
				long remaining = numSteps-i;
				long rtime = remaining*duration/i;//remaining/speed
				System.out.println( (rtime/(3600*1000)) + " hours remaining" );
			}
		}
		long end = System.currentTimeMillis();
		long duration = end-start;
		System.out.println( numSteps + " steps in " + duration + " millis" );
		System.out.println( ((1000.0*numSteps)/duration) + " steps per second" );
		
		System.out.println();
		env.simulationFinished();
		
		System.out.println( "Simulation finished" );
		
	}
	
}
